<?php
/* ===================== sales.php (Full POS Multi-Item) ===================== */
session_start();
include 'db.php';
include 'header.php';

// Fetch products and customers
$products = $conn->query("SELECT * FROM products ORDER BY name ASC");
$customers = $conn->query("SELECT * FROM customers ORDER BY name ASC");

// Handle Add Sale (Multi-Item)
if(isset($_POST['add'])){
    $customer_id = $_POST['customer_id'] ?: null;
    $sale_date = $_POST['sale_date'];
    $product_ids = $_POST['product_id'];
    $quantities = $_POST['quantity'];

    $error = '';

    $conn->begin_transaction();
    try {
        foreach($product_ids as $index => $product_id){
            $qty = (int)$quantities[$index];
            if($qty <= 0) continue;

            // Fetch product info
            $stmt = $conn->prepare("SELECT sale_price, cost_price, quantity FROM products WHERE id = ?");
            $stmt->bind_param("i", $product_id);
            $stmt->execute();
            $product = $stmt->get_result()->fetch_assoc();

            if(!$product) throw new Exception("Product not found.");
            if($qty > $product['quantity']) throw new Exception("Not enough stock for product ID $product_id");

            $unit_price = $product['sale_price'];
            $total = $unit_price * $qty;

            // Insert sale
            $stmt = $conn->prepare("INSERT INTO sales (customer_id, product_id, quantity, unit_price, total, sale_date) VALUES (?,?,?,?,?,?)");
            $stmt->bind_param("iiidds", $customer_id, $product_id, $qty, $unit_price, $total, $sale_date);
            $stmt->execute();

            // Update product stock
            $stmt = $conn->prepare("UPDATE products SET quantity = quantity - ? WHERE id = ?");
            $stmt->bind_param("ii", $qty, $product_id);
            $stmt->execute();
        }

        $conn->commit();
        header("Location: sales.php");
        exit;
    } catch(Exception $e){
        $conn->rollback();
        $error = $e->getMessage();
    }
}

// Handle Delete Sale
if(isset($_GET['delete'])){
    $id = (int)$_GET['delete'];
    $stmt = $conn->prepare("SELECT product_id, quantity FROM sales WHERE id = ?");
    $stmt->bind_param("i", $id);
    $stmt->execute();
    $sale_info = $stmt->get_result()->fetch_assoc();

    if($sale_info){
        $stmt = $conn->prepare("UPDATE products SET quantity = quantity + ? WHERE id = ?");
        $stmt->bind_param("ii", $sale_info['quantity'], $sale_info['product_id']);
        $stmt->execute();

        $stmt = $conn->prepare("DELETE FROM sales WHERE id = ?");
        $stmt->bind_param("i", $id);
        $stmt->execute();
    }
    header("Location: sales.php");
    exit;
}

// Search & Pagination
$search = $_GET['search'] ?? '';
$search_sql = $conn->real_escape_string($search);
$limit = 10;
$page = $_GET['page'] ?? 1;
$offset = ($page-1)*$limit;

$total_result = $conn->query("SELECT COUNT(*) as total FROM sales s JOIN products p ON s.product_id=p.id WHERE p.name LIKE '%$search_sql%'");
$total_row = $total_result->fetch_assoc();
$total = $total_row['total'];
$total_pages = ceil($total / $limit);

$sales = $conn->query("SELECT s.*, p.name as product_name, c.name as customer_name
                       FROM sales s
                       LEFT JOIN products p ON s.product_id=p.id
                       LEFT JOIN customers c ON s.customer_id=c.id
                       WHERE p.name LIKE '%$search_sql%'
                       ORDER BY s.sale_date DESC
                       LIMIT $offset,$limit");

// Calculate totals
$total_sales_amount = 0;
$total_profit = 0;
$all_sales = $conn->query("SELECT s.quantity, s.unit_price, p.cost_price FROM sales s JOIN products p ON s.product_id=p.id");
while($row = $all_sales->fetch_assoc()){
    $total_sales_amount += $row['unit_price'] * $row['quantity'];
    $total_profit += ($row['unit_price'] - $row['cost_price']) * $row['quantity'];
}
?>

<div class="container mt-4">
    <h3 class="mb-3">Sales-POS</h3>

    <?php if(isset($error) && $error): ?>
        <div class="alert alert-danger"><?= $error ?></div>
    <?php endif; ?>

    <!-- POS Multi-Item Sale Form -->
    <form method="POST" id="posForm" class="mb-4">
        <div class="row g-3 mb-2">
            <div class="col-md-4">
                <input list="customers" name="customer_id" class="form-control" placeholder="Select or type customer">
                <datalist id="customers">
                    <?php $customers->data_seek(0);
                    while($c = $customers->fetch_assoc()): ?>
                        <option value="<?= $c['id'] ?>"><?= htmlspecialchars($c['name']) ?></option>
                    <?php endwhile; ?>
                </datalist>
            </div>
            <div class="col-md-3">
                <input type="date" name="sale_date" class="form-control" value="<?= date('Y-m-d') ?>" required>
            </div>
        </div>

        <table class="table table-bordered" id="posTable">
            <thead class="table-dark">
                <tr>
                    <th>Product</th>
                    <th>Quantity</th>
                    <th>Unit Price</th>
                    <th>Total</th>
                    <th><button type="button" class="btn btn-success btn-sm" id="addRow">+</button></th>
                </tr>
            </thead>
            <tbody>
                <tr>
                    <td>
                        <select name="product_id[]" class="form-control product-select" required>
                            <option value="">Select Product</option>
                            <?php $products->data_seek(0);
                            while($p = $products->fetch_assoc()): ?>
                                <option value="<?= $p['id'] ?>" data-price="<?= $p['sale_price'] ?>"><?= htmlspecialchars($p['name']) ?> (Available: <?= $p['quantity'] ?>)</option>
                            <?php endwhile; ?>
                        </select>
                    </td>
                    <td><input type="number" name="quantity[]" class="form-control qty-input" min="1" value="1" required></td>
                    <td><input type="text" class="form-control unit-price" readonly></td>
                    <td><input type="text" class="form-control total-price" readonly></td>
                    <td></td>
                </tr>
            </tbody>
        </table>

        <button type="submit" name="add" class="btn btn-primary">Complete Sale</button>
    </form>

    <!-- Search Past Sales -->
    <form method="GET" class="row g-3 mb-3">
        <div class="col-md-6">
            <input type="text" name="search" class="form-control" placeholder="Search by Product Name" value="<?= htmlspecialchars($search) ?>">
        </div>
        <div class="col-md-2"><button class="btn btn-primary w-100">Search</button></div>
    </form>

    <!-- Export -->
    <a href="export_sales_excel.php" class="btn btn-success btn-sm mb-2">Export to Excel</a>
    <a href="export_sales_pdf.php" class="btn btn-danger btn-sm mb-2">Export to PDF</a>

    <!-- Sales Table -->
    <div class="table-responsive">
        <table class="table table-bordered">
            <thead class="table-dark">
                <tr>
                    <th>Customer</th>
                    <th>Product</th>
                    <th>Quantity</th>
                    <th>Unit Price</th>
                    <th>Total</th>
                    <th>Sale Date</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody>
                <?php while($row = $sales->fetch_assoc()): ?>
                <tr>
                    <td><?= htmlspecialchars($row['customer_name'] ?? '-') ?></td>
                    <td><?= htmlspecialchars($row['product_name']) ?></td>
                    <td><?= $row['quantity'] ?></td>
                    <td><?= number_format($row['unit_price'],2) ?></td>
                    <td><?= number_format($row['total'],2) ?></td>
                    <td><?= $row['sale_date'] ?></td>
                    <td> <?php if($_SESSION['role'] === 'admin'): ?>
                        <a href="sales_edit.php?id=<?= $row['id'] ?>" class="btn btn-warning btn-sm">Edit</a>
                        <a href="sales.php?delete=<?= $row['id'] ?>&search=<?= urlencode($search) ?>" class="btn btn-danger btn-sm" onclick="return confirm('Delete sale?')">Delete</a>
                                               <?php else: ?>
        <span class="text-muted"><center><i class="fa fa-lock"></i></center>
</span>
    <?php endif; ?>
                    </td>
                </tr>
                <?php endwhile; ?>
            </tbody>
        </table>
    </div>

    <!-- Pagination -->
    <nav>
        <ul class="pagination">
            <?php for($i=1; $i<=$total_pages; $i++): ?>
                <li class="page-item <?= ($i==$page)?'active':'' ?>">
                    <a class="page-link" href="?page=<?= $i ?>&search=<?= urlencode($search) ?>"><?= $i ?></a>
                </li>
            <?php endfor; ?>
        </ul>
    </nav>

    <p>Total Sales Amount: <?= number_format($total_sales_amount,2) ?></p>
    <p>Total Profit: <?= number_format($total_profit,2) ?></p>
</div>

<script>
// Dynamic POS table
document.getElementById('addRow').addEventListener('click', function(){
    let table = document.getElementById('posTable').getElementsByTagName('tbody')[0];
    let newRow = table.rows[0].cloneNode(true);
    newRow.querySelectorAll('input').forEach(i => i.value = '');
    newRow.querySelector('select').selectedIndex = 0;
    table.appendChild(newRow);
    updateUnitPrice();
});

function updateUnitPrice(){
    document.querySelectorAll('.product-select').forEach(function(select){
        select.addEventListener('change', function(){
            let price = parseFloat(this.selectedOptions[0].dataset.price) || 0;
            let row = this.closest('tr');
            row.querySelector('.unit-price').value = price.toFixed(2);
            row.querySelector('.total-price').value = (price * row.querySelector('.qty-input').value).toFixed(2);
        });
    });

    document.querySelectorAll('.qty-input').forEach(function(input){
        input.addEventListener('input', function(){
            let row = this.closest('tr');
            let price = parseFloat(row.querySelector('.unit-price').value) || 0;
            row.querySelector('.total-price').value = (price * this.value).toFixed(2);
        });
    });
}

updateUnitPrice();
</script>

<?php include 'footer.php'; ?>
